//
//  $Id: WXKEMail.m 108 2009-06-24 14:54:10Z fujidana $
//  Copyright 2006 FUJIDANA. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#import "WXKEMail.h"
#import "GSNSDataExtensions.h"
#include <string.h>


@interface WXKEMail (Private)

+ (NSDateFormatter *)sharedDateFormatter;
+ (NSString *)decodedStringInRFC2047MessageHeader:(NSString *)value;
- (void)calculateTransientValue;

@end


#pragma mark -


@implementation WXKEMail

- (void)awakeFromFetch
{
	[super awakeFromFetch];
}

- (NSData *)message
{
	NSData *tempObject;
	
    [self willAccessValueForKey:@"message"];
    tempObject = [self primitiveValueForKey:@"message"];
    [self didAccessValueForKey:@"message"];
	
    return tempObject;
}

- (void)setMessage:(NSData *)value
{
	[self willChangeValueForKey:@"message"];
	[self setPrimitiveValue:value forKey:@"message"];
	[self didChangeValueForKey:@"message"];
}

- (NSString *)body
{
	[self willAccessValueForKey:@"body"];
	NSString *value = [self primitiveValueForKey:@"body"];
	[self didAccessValueForKey:@"body"];
	
	if (value == nil)
	{
		[self calculateTransientValue];
		value = [self primitiveValueForKey:@"body"];
	}
	
	return value;
}

- (NSDate *)date
{
	[self willAccessValueForKey:@"date"];
	NSDate *value = [self primitiveValueForKey:@"date"];
	[self didAccessValueForKey:@"date"];
	
	if (value == nil)
	{
		[self calculateTransientValue];
		value = [self primitiveValueForKey:@"date"];
	}
	
	return value;
}

- (NSString *)from
{
	[self willAccessValueForKey:@"from"];
	NSString *value = [self primitiveValueForKey:@"from"];
	[self didAccessValueForKey:@"from"];
	
	if (value == nil)
	{
		[self calculateTransientValue];
		value = [self primitiveValueForKey:@"from"];
	}
	
	return value;
}

- (NSString *)subject
{
	[self willAccessValueForKey:@"subject"];
	NSString *value = [self primitiveValueForKey:@"subject"];
	[self didAccessValueForKey:@"subject"];
	
	if (value == nil)
	{
		[self calculateTransientValue];
		value = [self primitiveValueForKey:@"subject"];
	}
	
	return value;
}

- (NSString *)to
{
	[self willAccessValueForKey:@"to"];
	NSString *value = [self primitiveValueForKey:@"to"];
	[self didAccessValueForKey:@"to"];
	
	if (value == nil)
	{
		[self calculateTransientValue];
		value = [self primitiveValueForKey:@"to"];
	}
	
	return value;
}

- (NSString *)object
{
	if ([[self valueForKeyPath:@"container.order"] intValue] == WXKMailSentMailBox ||
		[[self valueForKeyPath:@"container.order"] intValue] == WXKMailDraftMailBox)
	{
		return [self valueForKey:@"to"];
	}
	else
	{
		return [self valueForKey:@"from"];
	}
}


@end


#pragma mark -


@implementation WXKEMail (Private)

+ (NSDateFormatter *)sharedDateFormatter
{
	static NSDateFormatter *sharedDateFormatter;
	
	if (sharedDateFormatter == nil)
	{
		NSLocale *POSIXLocale = [[[NSLocale alloc] initWithLocaleIdentifier:@"en_US_POSIX"] autorelease];
		
		sharedDateFormatter = [[NSDateFormatter alloc] init];
		[sharedDateFormatter setFormatterBehavior:NSDateFormatterBehavior10_4];
		[sharedDateFormatter setLocale:POSIXLocale];
		[sharedDateFormatter setDateFormat:@"EEE, d MMM yyyy HH:mm:ss Z"];
		[sharedDateFormatter setLenient:YES];
	}
	return sharedDateFormatter;
}

+ (NSString *)decodedStringInRFC2047MessageHeader:(NSString *)value
{
	NSScanner *scanner = [NSScanner scannerWithString:value];
	[scanner setCaseSensitive:NO];
	
	NSMutableString *result = [NSMutableString string];
	
	NSString *tempString;
	BOOL isScannerInEncodedString = NO;
	
	while ([scanner isAtEnd] == NO)
	{
		if (isScannerInEncodedString)
		{
			if ([scanner scanUpToString:@"?=" intoString:&tempString] &&
				[scanner scanString:@"?=" intoString:NULL])
			{
				NSArray *tempArray = [tempString componentsSeparatedByString:@"?"];
				if ([tempArray count] != 3)
				{
					NSLog(@"String does not agree with RFC2047 format.");
					return nil;
				}
				
				NSStringEncoding encoding = (NSStringEncoding)CFStringConvertEncodingToNSStringEncoding(CFStringConvertIANACharSetNameToEncoding((CFStringRef)[tempArray objectAtIndex:0]));
				if (encoding == 0)
				{
					NSLog(@"Following string encoding is not supported: %@", [tempArray objectAtIndex:0]);
					return nil;
				}
				
				NSString *decodedString;
				if ([[tempArray objectAtIndex:1] caseInsensitiveCompare:@"B"] == NSOrderedSame)
				{
					NSData *data = [NSData dataWithBase64EncodedString:[tempArray objectAtIndex:2]];
					if (data)
					{
						decodedString = [[[NSString alloc] initWithData:data
															   encoding:encoding] autorelease];
					}
				}
				else
				{
					NSLog(@"Q encoding is not supported");
					return nil;
				}
				if (decodedString)
				{
					[result appendString:decodedString];
				}
				isScannerInEncodedString = NO;
			}
			else
			{
				NSLog(@"ERROR! Not closed by ?=.");
				return nil;
			}
		}
		else
		{
			if ([scanner scanUpToString:@"=?" intoString:&tempString])
			{
				[result appendString:tempString];
			}
			if ([scanner scanString:@"=?" intoString:NULL])
			{
				isScannerInEncodedString = YES;
			}
		}
	}
	return [NSString stringWithString:result];
}


- (void)calculateTransientValue
{
	// Separate header and body at "\r\n\r\n"
	NSData *message = [self valueForKey:@"message"];
	
	int dataLength = [message length];
	
	char buffer[dataLength + 1];
	[message getBytes:buffer];
	buffer[dataLength] = '\0';
	
	char *splitPoint = strstr(buffer, "\r\n\r\n");
	if (splitPoint == NULL)
	{
		return;
	}
	
	int headerLength = (splitPoint + 2 - buffer); 
	
	NSData *headerData = [message subdataWithRange:NSMakeRange(0, headerLength)];
	NSData *bodyData = [message subdataWithRange:NSMakeRange(headerLength + 2, dataLength - (headerLength + 2))];
	NSMutableString *headerString = [[[NSMutableString alloc] initWithData:headerData encoding:NSShiftJISStringEncoding] autorelease];
	
	// In case failed to interpret text (probablly headerString contains EMOJI)
	if (headerString == nil)
	{
//		return NO;
		return;
	}
	[headerString replaceOccurrencesOfString:@"\r\n " withString:@" " options:NSLiteralSearch range:NSMakeRange(0, [headerString length])];
	[headerString replaceOccurrencesOfString:@"\r\n\t" withString:@"\t" options:NSLiteralSearch range:NSMakeRange(0, [headerString length])];
	
	NSScanner *headerScanner = [NSScanner scannerWithString:headerString];
	[headerScanner setCharactersToBeSkipped:[NSCharacterSet whitespaceCharacterSet]];
	NSString *returnString = @"\r\n";
	NSString *colonString = @":";
	NSString *type, *value;
	
	NSMutableDictionary *dict = [NSMutableDictionary dictionary];
	
	while ([headerScanner isAtEnd] == NO)
	{
		if ([headerScanner scanUpToString:colonString intoString:&type] && 
			[headerScanner scanString:colonString intoString:NULL])
		{
			if ([headerScanner scanString:returnString intoString:NULL])
			{
				value = @"";
			}
			else
			{
				[headerScanner scanUpToString:returnString intoString:&value];
				[headerScanner scanString:returnString intoString:NULL];
			}
			[dict setObject:value forKey:type];
		}
		else
		{
			break;
		}
	}
	
	NSString *tmpString;
	
	if ((tmpString = [dict objectForKey:@"Date"]) != nil)
	{
		NSDate *date = [[WXKEMail sharedDateFormatter] dateFromString:tmpString];
		[self setPrimitiveValue:date forKey:@"date"];
	}
	
	if ((tmpString = [dict objectForKey:@"Subject"]) != nil)
	{
		NSString *string = [WXKEMail decodedStringInRFC2047MessageHeader:tmpString];
		[self setPrimitiveValue:string forKey:@"subject"];
	}
	
	if ((tmpString = [dict objectForKey:@"To"]) != nil)
	{
		NSString *string = [WXKEMail decodedStringInRFC2047MessageHeader:tmpString];
		[self setPrimitiveValue:string forKey:@"to"];
	}
	
	if ((tmpString = [dict objectForKey:@"From"]) != nil)
	{
		NSString *string = [WXKEMail decodedStringInRFC2047MessageHeader:tmpString];
		[self setPrimitiveValue:string forKey:@"from"];
	}
	
	NSString *bodyString = [[[NSString alloc] initWithData:bodyData encoding:NSISO2022JPStringEncoding] autorelease];
	[self setPrimitiveValue:bodyString forKey:@"body"];
}

@end